<?php

declare(strict_types=1);

namespace Tfish\Model;

/**
 * \Tfish\Model\Yubikey class file.
 *
 * @copyright   Simon Wilkinson 2019+ (https://tuskfish.biz)
 * @license     https://www.gnu.org/licenses/old-licenses/gpl-2.0.en.html GNU General Public License (GPL) V2
 * @author      Simon Wilkinson <simon@isengard.biz>
 * @version     Release: 2.0
 * @since       2.0
 * @package     core
 */

/**
 * Model for two-factor login with Yubikey hardware token.
 *
 * @copyright   Simon Wilkinson 2019+ (https://tuskfish.biz)
 * @license     https://www.gnu.org/licenses/old-licenses/gpl-2.0.en.html GNU General Public License (GPL) V2
 * @author      Simon Wilkinson <simon@isengard.biz>
 * @version     Release: 2.0
 * @since       2.0
 * @package     core
 * @uses        trait \Tfish\Traits\ValidateString  Provides methods for validating UTF-8 character encoding and string composition.
 * @var         \Tfish\Session $session Instance of the session management class.
 */

class Yubikey
{
    use \Tfish\Traits\ValidateString;

    private $session;

    /**
     * Process admin login using two factor authentication (Yubikey hardware token).
     *
     * Note that you must obtain a Client ID from Yubico, and define it in trust_path/config.php.
     *
     * @param string $password
     * @param string $yubikeyOtp Yubikey one-time password, generated by a registered Yubikey.
     */
    public function login(string $password, string $yubikeyOtp)
    {
        $yubikeyOtp = $this->trimString($yubikeyOtp);
        $yubikey = new \Yubico\Auth_Yubico(TFISH_YUBIKEY_ID);

        $this->session->twoFactorLogin($password, $yubikeyOtp, $yubikey);
    }

    /**
     * Logout the user and redirect to the login form.
     */
    public function logout()
    {
        $this->session->logout(TFISH_URL . 'login/');
    }

    /**
     * Set the session object.
     *
     * This will be deprecated when the DICE dependency injection container is adopted.
     *
     * @param   \Tfish\Session $session Instance of the session management class.
     */
    public function setSession(\Tfish\Session $session)
    {
        $this->session = $session;
    }
}
